/*
* Copyright (C) 2004 Ladislav Vitasek
* info@wordrider.net
* http://www.wordrider.net
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package net.wordrider.dialogs.settings;

import net.wordrider.core.AppPrefs;
import net.wordrider.utilities.Swinger;

import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;
import javax.swing.text.PlainDocument;


/**
 * @author Vity
 */
class JTextFieldOption implements IOptionable<JTextField>, DocumentListener {
    private final String initValue;
    @SuppressWarnings({"fieldMayBeStatic"})
    private String defaultValue;
    private boolean wasChanged = false;
    private final OptionsGroupManager optionsGroupManager;
    private IOptionGroup group = null;
    private String propertyName = null;
    private JTextField field;


    public JTextFieldOption(final OptionsGroupManager optionsGroupManager, Document document, final String propertyName, final String defaultValue, final IOptionGroup group) {
        this(optionsGroupManager, document, AppPrefs.getProperty(propertyName, defaultValue), group);
        this.propertyName = propertyName;
        this.defaultValue = defaultValue;
    }

    private JTextFieldOption(final OptionsGroupManager optionsGroupManager, Document document, final String initValue, final IOptionGroup group) {
        field = new JTextField(document = (document == null ? new PlainDocument() : document), initValue, 0);
        this.initValue = initValue;
        this.group = group;
        this.optionsGroupManager = optionsGroupManager;
        document.addDocumentListener(this);
        Swinger.addKeyActions(this.field);
    }

    private void updateValue() {
        wasChanged = !(initValue.equals(this.field.getText()));
        optionsGroupManager.makeChange(this);
    }

    public void insertUpdate(DocumentEvent e) {
        updateValue();
    }

    public void removeUpdate(DocumentEvent e) {
        updateValue();
    }

    public void changedUpdate(DocumentEvent e) {
        updateValue();
    }


    public final void setDefault() {
        this.field.setText(this.defaultValue);
    }

    public final void restorePrevious() {
        this.field.setText(initValue);
    }

    public final boolean wasChanged() {
        return wasChanged;  //implement - call to super class
    }

    public void applyChange() {
        wasChanged = false;
        //applyedValue = this.getText();
        if (propertyName != null)
            AppPrefs.storeProperty(propertyName, this.field.getText());
    }


    public final IOptionGroup getOptionsGroup() {
        return group;  //implement - call to super class
    }

    public JTextField getComponent() {
        return field;
    }
}
